#ifndef __PDF_FUNCTIONS_H__
#define __PDF_FUNCTIONS_H__

#include "TF1.h"
#include "TH1F.h"
#include "TH2F.h"

#include <vector>


// ===================================================================
// Classi derivate da TF1 contenti le sezioni d'urto dsigma/dOmega
// - classe madre Xsection richiede di fornire un puntatore a funzione
//   double (*xsec)(double* x, double* p) dove:
//   x[0] è cos(theta)
//   p[0] è s
//   altri parametri dipendono dalla classe concreta.
// - classi derivate dovrebbero implementare xsec come static member 
//   function ed usarla nel costruttore. 
// - unità di misure: sqrt(s) in GeV
//                    dsdW (=dsigma/dOmega) in mb/sr
//                    sezione d'urto totale (intgrata su Omega) in mb
// - si suppone che la sezione d'urto sia simmetrica in azimuth

class Xsection : public TF1 {
 public:
  Xsection(const char* name,double (*xsec)(double*, double*), unsigned int npar, double sqrts=1.); 
  ~Xsection();
  void SetSqrts(double sqrts);
  double GetXsection(double sqrts, double cutoff=0.);
  double GetdsdW(double sqrts, double costheta);
  ClassDef(Xsection,1);
};

// ==============================================================
// Classi per fare liste di PDF e PDF+sezioni d'urto
// Utili se ad un certo processo possono partecipare più tipi di 
// partoni o più processi
 
class PDFlist {
public:
  PDFlist();
  PDFlist(const TF1 *fa,const TF1 *fb);
  virtual ~PDFlist();
  unsigned int AddPartonType(const TF1 *fa, const TF1 *fb);
  double GetWeight(double xa, double xb) const;
  TH1F* GetRapidityDistribution(double M, double sqrts) const;
  TH1F* GetPartonLuminosity(double sqrts) const;
 protected:
  std::vector<const TF1*> m_fa; 
  std::vector<const TF1*> m_fb; 
  ClassDef(PDFlist,1);
};

// ==============================================================
// Classe per il calcolo della sezione d'urto per un dato
// processo, con produzione di istogrammi relativi.

class XsecPDFlist {
 public:
  enum histnumber {RAPIDITY,SQRTSHAT,YVSHAT,Y1,Y2,Y1VSY2,DELTAY,PT1,PT2,PTVSY1,PTVSY2,MT,NMAX};
  XsecPDFlist();
  XsecPDFlist(const PDFlist *pdf,Xsection *xsec);
  virtual ~XsecPDFlist();
  unsigned int AddProcess(const PDFlist *pdf, Xsection *xsec);
  void ConfigureHisto(double sqrtsmin=10., double sqrtsmax=10000., double ymax=5);
  virtual double Simulate(double sqrts, unsigned int Nevents=1000000);
  void ShowHisto(histnumber i, const char* option="") const;
  TH1* GetHisto(histnumber i);
  double SetYRange(double ymax);
  double SetSRange(double sqrtsmin, double sqrtsmax);
  double GetXsec();
  double GetErrXsec();
  void PrintXsec();
 protected:
  void InitHisto();
  void ResetHisto();
  std::vector<const PDFlist*> m_pdf; 
  std::vector<Xsection*> m_xsec; 
  TH1* *m_histos;
  double m_ymax;
  double m_mmin;
  double m_mmax;
  double m_vxsec;
  double m_sxsec;
  ClassDef(XsecPDFlist,1);
};


// ==============================================================
// Simulazione di un esperimento, analoga alla classe XsecPDFlist
// ma ora permette di aggiungere delle risoluzioni sperimentali
// sul pT dei due oggetti e di avere gli istogrammi espressi 
// in numero di eventi per una certa luminostità.

class Experiment : public XsecPDFlist {
 public:
  Experiment();
  Experiment(const PDFlist *pdf,Xsection *xsec);
  virtual ~Experiment();
  virtual double Simulate(double sqrts, unsigned int Nevents=1000000);
  // Set integrated luminosity: histograms are normalized to show the number of of events;
  void SetIntegratedLumi(double intLumi); // integrated lumi in pb-1
  // Set pT resolution on object 1 and 2. The flag isRel state if resolution is relative
  // or absolute (in that case sigma_pt is expressed in GeV).
  void SetResolution(double sigma_pt1, bool isRel_1, double sigma_pt2, bool isRel_2);
 private:
  void ScaleForLumi(double factor);
  double m_lumi;
  double m_sigma_pt1;
  double m_sigma_pt2;
  bool m_isRel_1;
  bool m_isRel_2;
};

#endif
