#include "pdf_functions.h"

#include <cmath>
#include <iostream>
#include <cstdlib>

#include "TMath.h"
#include "TAxis.h"
#include "TRandom3.h"

// ===================================================================
// Set predefinito di funzioni di struttura,
// TF1 definite tra xMin (>0 per evitare singolarità 
// a piccolo x) e xMax (=1).
#include "constants.C"
#include "pdf_set1.C"

// ===== CLASS Xsection =======

Xsection::Xsection(const char* name,double (*xsec)(double*, double*), unsigned int npar, double sqrts) : 
  TF1(name,xsec,-1.,1.,npar) {
  if (npar<1) {
    std::cerr << "Error in building function " << name << ": at least one parameter is needed!" << std::endl;
    exit(-1);
  }
  SetSqrts(sqrts);
  GetYaxis()->SetTitle("d#sigma/d#Omega [mb]");
  GetXaxis()->SetTitle("cos$theta^{*}");
}

Xsection::~Xsection() { }

void Xsection::SetSqrts(double s) { SetParameter(0,s*s); }

double Xsection::GetXsection(double s, double cutoff) {
  double saveds=GetParameter(0);
  SetParameter(0,s*s);
  double xs = Integral(-1.+fabs(cutoff),1-fabs(cutoff));
  SetParameter(0,saveds);
  return TMath::TwoPi()*xs;
}

double Xsection::GetdsdW(double s, double costheta) {
  double saveds=GetParameter(0);
  SetParameter(0,s*s);
  double xs = Eval(costheta);
  SetParameter(0,saveds);
  return xs;
}


// ====== CLASS PDFlist =========


PDFlist::PDFlist() : m_fa(0), m_fb(0) { }

PDFlist::PDFlist(const TF1 *fa,const TF1 *fb) : m_fa(0), m_fb(0) { 
  AddPartonType(fa,fb);
}

PDFlist::~PDFlist() { }

unsigned int PDFlist::AddPartonType(const TF1 *fa, const TF1 *fb) {
  m_fa.push_back(fa);
  m_fb.push_back(fb);
  return m_fa.size();
} 

double PDFlist::GetWeight(double xa, double xb) const {
  double wgt=0.;
  if ( xa<xMin || xa>xMax || xb<xMin || xb>xMax ) return 0.; 
  for (unsigned int i=0; i<m_fa.size(); i++) {
    wgt += ( m_fa[i]->Eval(xa)*m_fb[i]->Eval(xb));
  }
  return wgt;
}

TH1F* PDFlist::GetRapidityDistribution(double M, double sqrts) const {
  double stau  = M/sqrts;
  double ymax = -log(stau);
  unsigned int nbins=100;
  TH1F* hy = new TH1F("pydistrib","y",nbins,-ymax,ymax);
  hy->GetXaxis()->SetTitle("y");
  hy->GetYaxis()->SetTitle("d^{2}N/dyd#tau");
  for (unsigned int i=1; i<=nbins; i++ ) {
    double y = hy->GetBinCenter(i);
    double expy= exp(y);
    hy->SetBinContent(i,GetWeight(stau*expy,stau/expy));
  }
  return hy;
}

TH1F* PDFlist::GetPartonLuminosity(double sqrts) const {
  // set histogram limits, from 10 GeV to 10 TeV, 
  // logarithmic, as in "Supercollider Physics"
  unsigned int nbins=100;
  double *xbins = new double[nbins+1];
  double lsmin = log (10.), lsmax = log(10000.);    
  double sl=(lsmax-lsmin)/nbins;
  for (unsigned int i=0; i<=nbins; i++) xbins[i]=exp(lsmin+i*sl); 
  TH1F *hL = new TH1F("pLumi","Parton Luminosity",nbins,xbins);
  hL->GetXaxis()->SetTitle("#sqrt{#hat{s}}");
  hL->GetYaxis()->SetTitle("dL/d#hat{s}");
  hL->GetYaxis()->SetRangeUser(1.E-6,1.E+6);
  hL->SetMinimum(1.E-6);
  hL->SetMaximum(1.E+6);
  double K = hc2*1.E6/(sqrts*sqrts); // parton luminosity in nb
  for (unsigned int i=1; i<=nbins; i++ ) {
    double M = hL->GetBinCenter(i);
    TH1F* h = GetRapidityDistribution(M,sqrts);
    hL->SetBinContent(i,h->Integral(1,h->GetNbinsX(),"width")*K);
    delete h;
  }
  return hL;
}

// ====== CLASS XsecPDFlist ===================

XsecPDFlist::XsecPDFlist() :
  m_pdf(0), m_xsec(0), m_ymax(5.), m_mmin(10.), m_mmax(10000.), m_vxsec(-1.), m_sxsec(-1.) 
{ 
  InitHisto(); 
}
 
XsecPDFlist::XsecPDFlist(const PDFlist *pdf,Xsection *xsec) :
  m_pdf(0), m_xsec(0), m_ymax(5.), m_mmin(10.), m_mmax(10000.), m_vxsec(-1.), m_sxsec(-1.) 
{ 
  AddProcess(pdf,xsec);
  InitHisto();
}

XsecPDFlist::~XsecPDFlist() { 
  for (unsigned int i=0; i<NMAX; i++) delete m_histos[i];
  delete[] m_histos;
}
  
unsigned int XsecPDFlist::AddProcess(const PDFlist *pdf, Xsection *xsec) {
  m_pdf.push_back(pdf);
  m_xsec.push_back(xsec);
  return m_pdf.size();
} 

double XsecPDFlist::Simulate(double sqrts, unsigned int Nevents) {
  ResetHisto();
  TRandom3 gen;
  double cutoff = 1.E-3;
  double taumin = pow(m_mmin/sqrts,2);
  double taumax = pow(m_mmax/sqrts,2);
  if (taumax>1.) taumax=1.;
  double volume=(taumax-taumin)*2.*(1.-cutoff)*TMath::TwoPi(); // normalizzazione senza limitazione in ymax
  double wgt_global=volume/Nevents; // il peso di partenza è il volume dello spazio di integrazione
  double xsec=0.;
  double xsec2=0.;
  for (unsigned int i=0; i<Nevents; i++) {
    double range=0.;
    /* 
     * integrazione in tau e poi in y
     */
    double tau = gen.Uniform(taumin,taumax);
    range=-log(tau);
    double y   = gen.Uniform(-0.5*range,0.5*range);
    if ( abs(y)>m_ymax ) continue; // point outside interesting range contribution=0;
    double ct  = gen.Uniform(-1.+cutoff,1.-cutoff);
    // variabili cinematiche
    double xa = sqrt(tau)*exp(y);
    double xb = sqrt(tau)/exp(y);
    double shat= sqrt(tau)*sqrts;
    double ys  = 0.5*log((1+ct)/(1-ct));
    double y1  = ys+y;
    double y2  =-ys+y;
    double pT  = 0.5*shat*sqrt(1-ct*ct);
    // peso di questo evento
    double wgt_event=0;
    for (unsigned int process=0; process<m_pdf.size(); process++) {
      double wgt1 = m_pdf[process]->GetWeight(xa,xb);
      double wgt2 = m_xsec[process]->GetdsdW(shat,ct);
      wgt_event += (wgt1*wgt2);
    }
    wgt_event*= (wgt_global*range);
    // riempimento degli istogrammi
    m_histos[RAPIDITY]->Fill(y,wgt_event);
    m_histos[SQRTSHAT]->Fill(shat,wgt_event);
    m_histos[      Y1]->Fill(y1,wgt_event);
    m_histos[      Y2]->Fill(y2,wgt_event);
    m_histos[  DELTAY]->Fill(y1-y2,wgt_event);
    m_histos[     PT1]->Fill(pT,wgt_event);
    m_histos[     PT2]->Fill(pT,wgt_event);
    m_histos[      MT]->Fill(2.*pT,wgt_event);
    dynamic_cast<TH2F*>(m_histos[  YVSHAT])->Fill(shat,y,wgt_event);
    dynamic_cast<TH2F*>(m_histos[  Y1VSY2])->Fill(y1,y2,wgt_event);
    dynamic_cast<TH2F*>(m_histos[  PTVSY1])->Fill(y1,pT,wgt_event);
    dynamic_cast<TH2F*>(m_histos[  PTVSY2])->Fill(y2,pT,wgt_event);
    xsec+=wgt_event;
    xsec2+=pow(wgt_event,2);
  }
  m_vxsec = xsec;
  m_sxsec = sqrt(xsec2-xsec*xsec/Nevents);
  return xsec;
}

void XsecPDFlist::InitHisto() {
  m_histos = new TH1*[NMAX];
  for (unsigned int i=0; i<NMAX; i++) m_histos[i]=0;
  ConfigureHisto(m_mmin,m_mmax,m_ymax);
}

void XsecPDFlist::ConfigureHisto(double sqrtsmin, double sqrtsmax, double ymax) {
  for (unsigned int i=0; i<NMAX; i++) {
    if ( m_histos[i] ) delete m_histos[i];
  }
  m_histos[RAPIDITY]=new TH1F("ydist","Rapidity",100,-ymax,ymax);
  m_histos[RAPIDITY]->GetXaxis()->SetTitle("y");
  m_histos[RAPIDITY]->GetYaxis()->SetTitle("#sigma [mb]");
  m_histos[SQRTSHAT]=new TH1F("shat","#sqrt{#hat{s}}",100,sqrtsmin,sqrtsmax);
  m_histos[SQRTSHAT]->GetXaxis()->SetTitle("M [GeV]");
  m_histos[SQRTSHAT]->GetYaxis()->SetTitle("#sigma [mb]");
  m_histos[      Y1]=new TH1F("y1","y_{1}",100,-ymax,ymax);
  m_histos[      Y1]->GetXaxis()->SetTitle("y");
  m_histos[      Y1]->GetYaxis()->SetTitle("#sigma [mb]");
  m_histos[      Y2]=new TH1F("y2","y_{2}",100,-ymax,ymax);
  m_histos[      Y2]->GetXaxis()->SetTitle("y");
  m_histos[      Y2]->GetYaxis()->SetTitle("#sigma [mb]");
  m_histos[  DELTAY]=new TH1F("ydiff","#Deltay_{1,2}",100,-ymax,ymax);
  m_histos[  DELTAY]->GetXaxis()->SetTitle("#Deltay");
  m_histos[  DELTAY]->GetYaxis()->SetTitle("#sigma [mb]");
  m_histos[     PT1]=new TH1F("pT1","p_{T,1}",100,0.,sqrtsmax/2.);
  m_histos[     PT1]->GetXaxis()->SetTitle("P_{T} [GeV]");
  m_histos[     PT1]->GetYaxis()->SetTitle("#sigma [mb]");
  m_histos[     PT2]=new TH1F("pT2","p_{T,2}",100,0.,sqrtsmax/2.);
  m_histos[     PT2]->GetXaxis()->SetTitle("P_{T} [GeV]");
  m_histos[     PT2]->GetYaxis()->SetTitle("#sigma [mb]");
  m_histos[      MT]=new TH1F("mT","m_{T}",100,0.,sqrtsmax);
  m_histos[      MT]->GetXaxis()->SetTitle("m_{T} [GeV]");
  m_histos[      MT]->GetYaxis()->SetTitle("#sigma [mb]");
  m_histos[  YVSHAT]=new TH2F("yvsshat","y vs. #sqrt{#hat{s}}",100,sqrtsmin,sqrtsmax,100,-ymax,ymax);
  m_histos[  YVSHAT]->GetXaxis()->SetTitle("M [GeV]");
  m_histos[  YVSHAT]->GetYaxis()->SetTitle("y");
  m_histos[  YVSHAT]->GetZaxis()->SetTitle("#sigma [mb]");
  m_histos[  Y1VSY2]=new TH2F("y1vsy2","y_{2} vs. y_{1}",100,-ymax,ymax,100,-ymax,ymax);
  m_histos[  Y1VSY2]->GetXaxis()->SetTitle("y_{2}");
  m_histos[  Y1VSY2]->GetYaxis()->SetTitle("y_{1}");
  m_histos[  Y1VSY2]->GetZaxis()->SetTitle("#sigma [mb]");
  m_histos[  PTVSY1]=new TH2F("ptvsy1","p_{T} vs. y_{1}",100,-ymax,ymax,100,0.,sqrtsmax/2.);
  m_histos[  PTVSY1]->GetXaxis()->SetTitle("y");
  m_histos[  PTVSY1]->GetYaxis()->SetTitle("P_{T} [GeV]");
  m_histos[  PTVSY1]->GetZaxis()->SetTitle("#sigma [mb]");
  m_histos[  PTVSY2]=new TH2F("ptvsy2","p_{T} vs. y_{2}",100,-ymax,ymax,100,0.,sqrtsmax/2.);
  m_histos[  PTVSY2]->GetXaxis()->SetTitle("y");
  m_histos[  PTVSY2]->GetYaxis()->SetTitle("P_{T} [GeV]");
  m_histos[  PTVSY2]->GetZaxis()->SetTitle("#sigma [mb]");
}

void XsecPDFlist::ResetHisto() {
  for (unsigned int i=0; i<NMAX; i++) m_histos[i]->Reset(); 
}

void XsecPDFlist::ShowHisto(histnumber i, const char* option) const {
  m_histos[i]->Draw(option);
}

TH1* XsecPDFlist::GetHisto(histnumber i) { 
  return m_histos[i];
}

double XsecPDFlist::SetYRange(double ymax) {
  m_ymax=ymax;
  ConfigureHisto(m_mmin,m_mmax,m_ymax);
  return m_ymax;
}

double XsecPDFlist::SetSRange(double sqrtsmin, double sqrtsmax) {
  m_mmin = sqrtsmin;
  m_mmax = sqrtsmax;
  ConfigureHisto(m_mmin,m_mmax,m_ymax);
  return m_mmax-m_mmin;
}
 
double XsecPDFlist::GetXsec() {
  return m_vxsec;
}

double XsecPDFlist::GetErrXsec() {
  return m_sxsec;
}

void XsecPDFlist::PrintXsec() {
  cout << "sigma(" << m_mmin << "< sqrt{s} < " << m_mmax << "GeV ) = " << m_vxsec << " +- " << m_sxsec << " mb" << endl; 
}

// ====== CLASS Experimento ===================

Experiment::Experiment() :
  XsecPDFlist(), m_lumi(-1.), m_sigma_pt1(0.), m_sigma_pt2(0.), m_isRel_1(false), m_isRel_2(false)
{ 

}
 
Experiment::Experiment(const PDFlist *pdf,Xsection *xsec) :
  XsecPDFlist(pdf, xsec), m_lumi(-1.), m_sigma_pt1(0.), m_sigma_pt2(0.), m_isRel_1(false), m_isRel_2(false)
{ 

}

Experiment::~Experiment() { 
  XsecPDFlist::~XsecPDFlist();
}
  
double Experiment::Simulate(double sqrts, unsigned int Nevents) {
  ResetHisto();
  TRandom3 gen;
  double cutoff = 1.E-3;
  double taumin = pow(m_mmin/sqrts,2);
  double taumax = pow(m_mmax/sqrts,2);
  if (taumax>1.) taumax=1.;
  double volume=(taumax-taumin)*2.*(1.-cutoff)*TMath::TwoPi(); // normalizzazione senza limitazione in ymax
  double wgt_global=volume/Nevents; // il peso di partenza è il volume dello spazio di integrazione
  double xsec=0.;
  double xsec2=0.;
  for (unsigned int i=0; i<Nevents; i++) {
    double range=0.;
    /* 
     * integrazione in tau e poi in y
     */
    double tau = gen.Uniform(taumin,taumax);
    range=-log(tau);
    double y   = gen.Uniform(-0.5*range,0.5*range);
    if ( abs(y)>m_ymax ) continue; // point outside interesting range contribution=0;
    double ct  = gen.Uniform(-1.+cutoff,1.-cutoff);
    // variabili cinematiche
    double xa = sqrt(tau)*exp(y);
    double xb = sqrt(tau)/exp(y);
    double shat= sqrt(tau)*sqrts;
    double ys  = 0.5*log((1+ct)/(1-ct));
    double y1  = ys+y;
    double y2  =-ys+y;
    double pT  = 0.5*shat*sqrt(1-ct*ct);
    double pT1 = gen.Gaus(pT, (m_isRel_1?m_sigma_pt1*pT:m_sigma_pt1) );
    double pT2 = gen.Gaus(pT, (m_isRel_2?m_sigma_pt2*pT:m_sigma_pt2) );
    double mT  = sqrt(4.*pT1*pT2);
    // peso di questo evento
    double wgt_event=0;
    for (unsigned int process=0; process<m_pdf.size(); process++) {
      double wgt1 = m_pdf[process]->GetWeight(xa,xb);
      double wgt2 = m_xsec[process]->GetdsdW(shat,ct);
      wgt_event += (wgt1*wgt2);
    }
    wgt_event*= (wgt_global*range);
    // riempimento degli istogrammi
    m_histos[RAPIDITY]->Fill(y,wgt_event);
    m_histos[SQRTSHAT]->Fill(shat,wgt_event);
    m_histos[      Y1]->Fill(y1,wgt_event);
    m_histos[      Y2]->Fill(y2,wgt_event);
    m_histos[  DELTAY]->Fill(y1-y2,wgt_event);
    m_histos[     PT1]->Fill(pT1,wgt_event);
    m_histos[     PT2]->Fill(pT2,wgt_event);
    m_histos[      MT]->Fill(mT,wgt_event);
    dynamic_cast<TH2F*>(m_histos[  YVSHAT])->Fill(shat,y,wgt_event);
    dynamic_cast<TH2F*>(m_histos[  Y1VSY2])->Fill(y1,y2,wgt_event);
    dynamic_cast<TH2F*>(m_histos[  PTVSY1])->Fill(y1,pT1,wgt_event);
    dynamic_cast<TH2F*>(m_histos[  PTVSY2])->Fill(y2,pT2,wgt_event);
    xsec+=wgt_event;
    xsec2+=pow(wgt_event,2);
  }
  m_vxsec = xsec;
  m_sxsec = sqrt(xsec2-xsec*xsec/Nevents);
  if (m_lumi>=0) ScaleForLumi(m_lumi*(mb/pb));
  return xsec;
}

void Experiment::SetIntegratedLumi(double intLumi) {
  double factor = intLumi*(mb/pb);
  if ( m_lumi>0 ) factor=intLumi/m_lumi;
  ScaleForLumi(factor);
  m_lumi = intLumi;
}

void Experiment::SetResolution(double sigma_pt1, bool isRel_1, double sigma_pt2, bool isRel_2) {
  m_sigma_pt1 = sigma_pt1;
  m_isRel_1   = isRel_1;
  m_sigma_pt2 = sigma_pt2;
  m_isRel_2   = isRel_2;
}

void Experiment::ScaleForLumi(double factor) {
  for (unsigned int i=0; i<NMAX; i++) {
    m_histos[i]->Scale(factor);
    if ( dynamic_cast<TH2F*>(m_histos[i]) ) m_histos[i]->GetZaxis()->SetTitle("Events");
    else  m_histos[i]->GetYaxis()->SetTitle("Events");
  } 
}
